/*
 * cph1_rev.c
 * By Matthew Skala
 */

#include <stdio.h>

/* Polynomial table for forward CRC */
unsigned long crctable[256]={
   0x00000000L, 0x77073096L, 0xEE0E612CL, 0x990951BAL,
   0x076DC419L, 0x706AF48FL, 0xE963A535L, 0x9E6495A3L,
   0x0EDB8832L, 0x79DCB8A4L, 0xE0D5E91EL, 0x97D2D988L,
   0x09B64C2BL, 0x7EB17CBDL, 0xE7B82D07L, 0x90BF1D91L,
   0x1DB71064L, 0x6AB020F2L, 0xF3B97148L, 0x84BE41DEL,
   0x1ADAD47DL, 0x6DDDE4EBL, 0xF4D4B551L, 0x83D385C7L,
   0x136C9856L, 0x646BA8C0L, 0xFD62F97AL, 0x8A65C9ECL,
   0x14015C4FL, 0x63066CD9L, 0xFA0F3D63L, 0x8D080DF5L,
   0x3B6E20C8L, 0x4C69105EL, 0xD56041E4L, 0xA2677172L,
   0x3C03E4D1L, 0x4B04D447L, 0xD20D85FDL, 0xA50AB56BL,
   0x35B5A8FAL, 0x42B2986CL, 0xDBBBC9D6L, 0xACBCF940L,
   0x32D86CE3L, 0x45DF5C75L, 0xDCD60DCFL, 0xABD13D59L,
   0x26D930ACL, 0x51DE003AL, 0xC8D75180L, 0xBFD06116L,
   0x21B4F4B5L, 0x56B3C423L, 0xCFBA9599L, 0xB8BDA50FL,
   0x2802B89EL, 0x5F058808L, 0xC60CD9B2L, 0xB10BE924L,
   0x2F6F7C87L, 0x58684C11L, 0xC1611DABL, 0xB6662D3DL,
   0x76DC4190L, 0x01DB7106L, 0x98D220BCL, 0xEFD5102AL,
   0x71B18589L, 0x06B6B51FL, 0x9FBFE4A5L, 0xE8B8D433L,
   0x7807C9A2L, 0x0F00F934L, 0x9609A88EL, 0xE10E9818L,
   0x7F6A0DBBL, 0x086D3D2DL, 0x91646C97L, 0xE6635C01L,
   0x6B6B51F4L, 0x1C6C6162L, 0x856530D8L, 0xF262004EL,
   0x6C0695EDL, 0x1B01A57BL, 0x8208F4C1L, 0xF50FC457L,
   0x65B0D9C6L, 0x12B7E950L, 0x8BBEB8EAL, 0xFCB9887CL,
   0x62DD1DDFL, 0x15DA2D49L, 0x8CD37CF3L, 0xFBD44C65L,
   0x4DB26158L, 0x3AB551CEL, 0xA3BC0074L, 0xD4BB30E2L,
   0x4ADFA541L, 0x3DD895D7L, 0xA4D1C46DL, 0xD3D6F4FBL,
   0x4369E96AL, 0x346ED9FCL, 0xAD678846L, 0xDA60B8D0L,
   0x44042D73L, 0x33031DE5L, 0xAA0A4C5FL, 0xDD0D7CC9L,
   0x5005713CL, 0x270241AAL, 0xBE0B1010L, 0xC90C2086L,
   0x5768B525L, 0x206F85B3L, 0xB966D409L, 0xCE61E49FL,
   0x5EDEF90EL, 0x29D9C998L, 0xB0D09822L, 0xC7D7A8B4L,
   0x59B33D17L, 0x2EB40D81L, 0xB7BD5C3BL, 0xC0BA6CADL,
   0xEDB88320L, 0x9ABFB3B6L, 0x03B6E20CL, 0x74B1D29AL,
   0xEAD54739L, 0x9DD277AFL, 0x04DB2615L, 0x73DC1683L,
   0xE3630B12L, 0x94643B84L, 0x0D6D6A3EL, 0x7A6A5AA8L,
   0xE40ECF0BL, 0x9309FF9DL, 0x0A00AE27L, 0x7D079EB1L,
   0xF00F9344L, 0x8708A3D2L, 0x1E01F268L, 0x6906C2FEL,
   0xF762575DL, 0x806567CBL, 0x196C3671L, 0x6E6B06E7L,
   0xFED41B76L, 0x89D32BE0L, 0x10DA7A5AL, 0x67DD4ACCL,
   0xF9B9DF6FL, 0x8EBEEFF9L, 0x17B7BE43L, 0x60B08ED5L,
   0xD6D6A3E8L, 0xA1D1937EL, 0x38D8C2C4L, 0x4FDFF252L,
   0xD1BB67F1L, 0xA6BC5767L, 0x3FB506DDL, 0x48B2364BL,
   0xD80D2BDAL, 0xAF0A1B4CL, 0x36034AF6L, 0x41047A60L,
   0xDF60EFC3L, 0xA867DF55L, 0x316E8EEFL, 0x4669BE79L,
   0xCB61B38CL, 0xBC66831AL, 0x256FD2A0L, 0x5268E236L,
   0xCC0C7795L, 0xBB0B4703L, 0x220216B9L, 0x5505262FL,
   0xC5BA3BBEL, 0xB2BD0B28L, 0x2BB45A92L, 0x5CB36A04L,
   0xC2D7FFA7L, 0xB5D0CF31L, 0x2CD99E8BL, 0x5BDEAE1DL,
   0x9B64C2B0L, 0xEC63F226L, 0x756AA39CL, 0x026D930AL,
   0x9C0906A9L, 0xEB0E363FL, 0x72076785L, 0x05005713L,
   0x95BF4A82L, 0xE2B87A14L, 0x7BB12BAEL, 0x0CB61B38L,
   0x92D28E9BL, 0xE5D5BE0DL, 0x7CDCEFB7L, 0x0BDBDF21L,
   0x86D3D2D4L, 0xF1D4E242L, 0x68DDB3F8L, 0x1FDA836EL,
   0x81BE16CDL, 0xF6B9265BL, 0x6FB077E1L, 0x18B74777L,
   0x88085AE6L, 0xFF0F6A70L, 0x66063BCAL, 0x11010B5CL,
   0x8F659EFFL, 0xF862AE69L, 0x616BFFD3L, 0x166CCF45L,
   0xA00AE278L, 0xD70DD2EEL, 0x4E048354L, 0x3903B3C2L,
   0xA7672661L, 0xD06016F7L, 0x4969474DL, 0x3E6E77DBL,
   0xAED16A4AL, 0xD9D65ADCL, 0x40DF0B66L, 0x37D83BF0L,
   0xA9BCAE53L, 0xDEBB9EC5L, 0x47B2CF7FL, 0x30B5FFE9L,
   0xBDBDF21CL, 0xCABAC28AL, 0x53B39330L, 0x24B4A3A6L,
   0xBAD03605L, 0xCDD70693L, 0x54DE5729L, 0x23D967BFL,
   0xB3667A2EL, 0xC4614AB8L, 0x5D681B02L, 0x2A6F2B94L,
   0xB40BBE37L, 0xC30C8EA1L, 0x5A05DF1BL, 0x2D02EF8DL
};

/* This answers the question: how many freely-chosen bits do I include
 * when I ask for a crc collision with input length (index)? */
char freebits[12]={0,0,0,0,1,4,10,16,22,28,34,40};

/* This says where each forced bit goes */
char bitsforced[32]={0,1,2,3,4,6,
                     8,9,10,11,12,14,
                     16,17,18,19,20,22,
                     24,25,26,27,28,30,
                     32,33,34,35,38,
                     40,41,42};

/* This says where each free bit goes */
char bitsfree[40]={36,43,44,46,
                   48,49,50,51,52,54,
                   56,57,58,59,60,62,
                   64,65,66,67,68,70,
                   72,73,74,75,76,78,
                   80,81,82,83,84,86,
                   88,89,90,91,92,94};

/* The portion of the inverted matrix corresponding to the CRC bits */
unsigned long crcmatrix[32]={
   0x9BF7B4FE,0x10CEBBDB,0x3EC28E73,0xE516F5B2,
   0x3EB07172,0xAC6CB91B,0x2344667F,0x25ECE58C,
   0xD24109C4,0x501CB10A,0x97761211,0x0A2EF700,
   0x0C806D13,0x55AE3901,0x4C147270,0xDAC3C857,
   0x384B8A54,0xF7583CAD,0xA1DA1DC4,0x0028BBDC,
   0xB5BB7FE3,0x99610C1A,0x1FC446C4,0x8DE0FF05,
   0x01D3D128,0x64FAC9B2,0x3BC5E604,0xE564A85C,
   0xADEB84A5,0xCFCDBB2B,0x3E7D9F68,0xA102B971
};

/* The portion of the inverted matrix corresponding to the free bits */
unsigned long freematrix[40]={
   0x0CBFC054,0xAEAB35B2,0x315B20B2,0x1F113696,
   0x6DA65FB4,0x08F3CFCD,0xC0E8FCF1,0xD928FA77,
   0x58C085F6,0x55F7A6A4,0x726948CB,0xBEE706A6,
   0xDE9BCF28,0x539FADD8,0xA5D7713D,0xA6B4900F,
   0x3CA9547B,0xC98AC9B5,0xAF52FA18,0x60098F5B,
   0x142D2C51,0x706AA085,0x46494250,0x54026BCE,
   0xEBE4D0A3,0x673646B9,0x945A22D6,0x7C5347FB,
   0xC61C9B99,0x97780ADB,0x7E9DB1AE,0x88C43E39,
   0x55CEBFB3,0x5C81ADC9,0x0F3DD57C,0x3D44BCF3,
   0x0383F8DD,0x73F38757,0xA8F2D5CF,0x2922BEA9
};

/* Matrix columns to take into account the canonicalization */
unsigned long lengthmatrix[12]={
   0x84741063,0xC5273406,0xE5A222DF,0x9941CB2B,
   0xD9EBE522,0xCB93A8AF,0x962E3D2D,0x90029144,
   0x5B298B04,0x575F1D8A,0x78EE4BEC,0x47B6B86A
};

/* Macros to get and flip bits.  FLIPBITR does the special reverse indexing
 * mandated by the way we numbered our bits. */
#define GETBIT(p,b) ((((p)[(b)>>3])>>((b)&7))&1)
#define FLIPBIT(p,b) ((p)[(b)>>3]^=(1<<((b)&7)))
#define FLIPBITR(p,b) { if (length-1-((b)>>3)>=0) \
                        (p)[length-1-((b)>>3)]^=(1<<((b)&7)); }

/* Attempt to reverse CRC32, given the desired CRC value, the length, and
 * some choices for the "free" bits. */
void reverse_crc(unsigned long crc,int length,char *in,char *out) {
   unsigned long bits;
   int i;
   
   /* correct for output length */
   bits=lengthmatrix[length-1];
   
   /* XOR in the CRC */
   for (i=0;i<32;i++)
     if (crc&(1<<i))
       bits^=crcmatrix[i];
   
   /* XOR in the free bits */
   for (i=0;i<freebits[length-1];i++)
     if (GETBIT(in,i))
       bits^=freematrix[i];
   
   /* set up output */
   for (i=0;i<length;i++)
     out[i]=0x20;
   
   /* output forced bits */
   for (i=0;i<32;i++)
     if (bits&(1<<i))
       FLIPBITR(out,bitsforced[i]);
   
   /* output free bits */
   for (i=0;i<freebits[length-1];i++)
     if (GETBIT(in,i))
       FLIPBITR(out,bitsfree[i]);
}

/* The CP4 HQ password hash function */
void cp_hash(char *input,int length,
	     unsigned long *hash1,unsigned long *hash2) {
   int i;
   
   *hash1=0;
   *hash2=0;
   for (i=0;i<length;i++) {
      *hash1=(*hash1>>8)^crctable[(*hash1&0xFF)^(input[i]|0x20)];
      *hash2=(*hash2<<5)+(*hash2>>27)+(input[i]|0x20)-*hash1;
   }
}

/* main program.  To use, just type in the 16-digit hex string as seen in the
 * ini file. */
int main(void) {
   unsigned long hash1,hash2,gothash1,gothash2;
   int length,bflip,i;
   char freeb[6],plaintext[12];
   
   scanf("%08x%08x",&hash1,&hash2);
   
   for (length=1;length<=12;length++) {
      printf("Trying length %d (2^%d possibilities)...\n",
	     length,freebits[length-1]);
      for (i=0;i<6;i++) freeb[i]=0;
      
      do {
	 reverse_crc(hash1,length,freeb,plaintext);
	 cp_hash(plaintext,length,&gothash1,&gothash2);
	 
	 if ((hash1==gothash1) && (hash2==gothash2)) {
	    printf("Found: \"");
	    for (i=0;i<length;i++)
	      putchar(plaintext[i]);
	    printf("\"\n");
	    return 0;
	 }
	 
	 for (bflip=0;GETBIT(freeb,bflip);bflip++) FLIPBIT(freeb,bflip);
	 FLIPBIT(freeb,bflip);
      } while (bflip<freebits[length-1]);
   }

   return 1;
}
